<?php

namespace Modules\Property\Controllers;

use App\Http\Controllers\Controller;
use Modules\Tracking\Facades\Tracker;
use Modules\Property\Models\Property;
use Illuminate\Http\Request;
use Modules\Location\Models\Location;
use Modules\Review\Models\Review;
use Modules\Core\Models\Attributes;
use DB;
use Modules\Property\Models\PropertyCategory;
use Modules\Property\ModuleProvider;

class PropertyController extends Controller
{
    protected $propertyClass;
    protected $locationClass;
    protected $propertyCategoryClass;
    protected $attributeClass;
    protected $reviewClass;

    public function __construct(Property $propertyClass)
    {
        $this->locationClass = Location::class;
        $this->propertyCategoryClass = PropertyCategory::class;
        $this->attributeClass = Attributes::class;
        $this->reviewClass = Review::class;
        $this->propertyClass = $propertyClass;
    }

    public function callAction($method, $parameters)
    {
        if (!Property::isEnable()) {
            return redirect('/');
        }
        return parent::callAction($method, $parameters); // TODO: Change the autogenerated stub
    }

    public function index(Request $request)
    {
        $is_ajax = $request->query('_ajax');
        $layout = setting_item("property_layout_search", 'normal');
        if ($request->query('_layout')) {
            $layout = $request->query('_layout');
        }
        $grids = request('_grids', setting_item('grids', 2));
        $_display = $request->query('_display', setting_item('property_display', 'grid'));
        if (!in_array($_display, ['list', 'grid'])) $_display = 'grid';

        $for_map = $request->query('_map', $layout === 'map');
        $query = $this->propertyClass->search($request);
        $list = $query->paginate(15);
        $markers = [];
        $for_map = $for_map ?: in_array($layout, config('property.for_map_layouts'));
        if (!empty($list) and $for_map) {
            foreach ($list as $row) {
                $markers[] = [
                    "id"      => $row->id,
                    "title"   => $row->title,
                    "lat"     => (float)$row->map_lat,
                    "lng"     => (float)$row->map_lng,
                    "gallery" => $row->getGallery(true),
                    "infobox" => view('Property::frontend.layouts.search.loop-grid', ['row'              => $row,
                                                                                      'disable_lazyload' => 1,
                                                                                      'wrap_class'       => 'infobox-item',
                                                                                      'display'          => 'grid'])->render(),
                    'marker'  => url('images/icons/png/pin.png'),
                ];
            }
        }
        if ($is_ajax) {
            return [
                'fragments' => [
                    '.ajax-search-result' => view('Property::frontend.ajax.search-result', [
                        'rows'    => $list,
                        'layout'  => $layout,
                        'display' => $_display,
                        'grids'   => $grids
                    ])->render(),
                    '.result-count'       => __("Showing :from - :to of :total properties", ["from"  => $list->firstItem(),
                                                                                             "to"    => $list->lastItem(),
                                                                                             "total" => $list->total()]),
                ],
                'markers'   => $markers
            ];
        }
        $data = [
            'rows'                   => $list,
            'list_location'          => $this->locationClass::where('status', 'publish')->with(['translation'])->get()->toTree(),
            'list_category'          => $this->propertyCategoryClass::where('status', 'publish')->get()->toTree(),
            'property_min_max_price' => $this->propertyClass::getMinMaxPrice(),
            'markers'                => $markers,
            "blank"                  => 1,
            "filter"                 => $request->query('filter'),
            "seo_meta"               => $this->propertyClass::getSeoMetaForPageList(),
            'display'                => $_display,
            'layout'                 => $layout
        ];
        $data['attributes'] = $this->attributeClass::where('service', 'property')->with(['terms', 'translation'])->get();

        $data['property_layout_map_option'] = setting_item_with_lang('property_layout_map_option', app()->getLocale(), $request->query('layout_map', 'map_left'));
        $data['property_layout_map_size'] = setting_item_with_lang('property_layout_map_size', app()->getLocale(), $request->query('layout_map_size', 4));

        if ($layout == "map") {
            $data['body_class'] = 'has-search-map';
            $data['html_class'] = 'full-page';
            return view('Property::frontend.search-map', $data);
        }
        $layout_id = $request->input('layout') ? $request->input('layout') : setting_item("property_page_search_layout", 1);
        $data['view'] = 'Property::frontend.layouts.search.list-item-rtl';
        if ($layout_id == 1) {
            $data['view'] = 'Property::frontend.layouts.search.list-item-rtl';
        } elseif ($layout_id == 2) {
            $data['view'] = 'Property::frontend.layouts.search.list-item-ltr';
        }
        return view('Property::frontend.search', $data);
    }

    public function detail(Request $request, $slug)
    {

        $layout_id = $request->input('layout') ? $request->input('layout') : setting_item("property_page_single_layout", 1);
        $limit_location = 15;
        if (empty(setting_item("property_location_search_style")) or setting_item("property_location_search_style") == "normal") {
            $limit_location = 1000;
        }
        $row = $this->propertyClass::where('slug', $slug)->with(['location', 'translation', 'hasWishList'])->first();
        if (empty($row) or !$row->hasPermissionDetailView()) {
            return redirect('/');
        }
        Tracker::track(Tracker::PAGE_VIEW, [
            'object_id'    => $row->id,
            'object_model' => 'property',
            'vendor_id'    => $row->author_id
        ]);
        $translation = $row->translate();
        $property_related = [];
        $location_id = $row->location_id;
        if (!empty($location_id)) {
            $property_related = $this->propertyClass::where('location_id', $location_id)->where("status", "publish")->take(4)->whereNotIn('id', [$row->id])->with(['location', 'translation', 'hasWishList'])->get();
        }
        $review_list = Review::where('object_id', $row->id)->where('object_model', 'property')->where("status", "approved")->orderBy("id", "desc")->with('author')->paginate(setting_item('property_review_number_per_page', 5));
        $row->view = $row->view + 1;
        $row->save();

        $data = [
            'row'                    => $row,
            'translation'            => $translation,
            'property_related'       => $property_related,
            'booking_data'           => $row->getBookingData(),
            'list_location'          => $this->locationClass::where('status', 'publish')->with(['translation'])->get()->toTree(),
            'list_category'          => $this->propertyCategoryClass::where('status', 'publish')->get()->toTree(),
            'property_min_max_price' => $this->propertyClass::getMinMaxPrice(),
            'review_list'            => $review_list,
            'seo_meta'               => $row->getSeoMetaWithTranslation(app()->getLocale(), $translation),
            'body_class'             => 'is_single'
        ];
        $this->setActiveMenu($row);
        $blade = 'Property::frontend.detail';
        if ($layout_id == 1) {
            $blade = 'Property::frontend.detail';
        } elseif ($layout_id == 2) {
            $blade = 'Property::frontend.detail_v2';
        } elseif ($layout_id == 3) {
            $blade = 'Property::frontend.detail_v3';
        } elseif ($layout_id == 4) {
            $blade = 'Property::frontend.detail_v4';
        }

        return view($blade, $data);
    }
}
