<?php
namespace Modules\Course\Controllers;

use Modules\Course\Models\Course;
use Modules\Course\Models\CourseTerm;
use Modules\Course\Models\CourseTranslation;
use Modules\FrontendController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\Location\Models\Location;
use Modules\Core\Models\Attributes;
use Modules\Booking\Models\Booking;

class ManageCourseController extends FrontendController
{
    protected $courseClass;
    protected $jobTranslationClass;
    protected $jobTermClass;
    protected $attributesClass;
    protected $locationClass;
    protected $bookingClass;
    public function __construct()
    {
        parent::__construct();
        $this->courseClass = Course::class;
        $this->jobTranslationClass = CourseTranslation::class;
        $this->jobTermClass = CourseTerm::class;
        $this->attributesClass = Attributes::class;
        $this->locationClass = Location::class;
        $this->bookingClass = Booking::class;
    }

    public function callAction($method, $parameters)
    {
        if(!Course::isEnable())
        {
            return redirect('/');
        }
        return parent::callAction($method, $parameters); // TODO: Change the autogenerated stub
    }
    public function manageCourse(Request $request)
    {
        $this->checkPermission('course_view');
        $user_id = Auth::id();
        $list_tour = $this->courseClass::where("create_user", $user_id)->orderBy('id', 'desc');
        $data = [
            'rows' => $list_tour->paginate(5),
            'breadcrumbs'        => [
                [
                    'name' => __('Manage Courses'),
                    'url'  => route('course.vendor.index')
                ],
                [
                    'name'  => __('All'),
                    'class' => 'active'
                ],
            ],
            'page_title'         => __("Manage Courses"),
        ];
        return view('Course::frontend.manageCourse.index', $data);
    }

    public function createCourse(Request $request)
    {
        $this->checkPermission('tour_create');
        $row = new $this->courseClass();
        $data = [
            'row'           => $row,
            'translation' => new $this->jobTranslationClass(),
            'course_location' => $this->locationClass::get()->toTree(),
            'attributes'    => $this->attributesClass::where('service', 'course')->get(),
            'breadcrumbs'        => [
                [
                    'name' => __('Manage Courses'),
                    'url'  => route('course.vendor.index')
                ],
                [
                    'name'  => __('Create'),
                    'class' => 'active'
                ],
            ],
            'page_title'         => __("Create Courses"),
        ];
        return view('Course::frontend.manageCourse.detail', $data);
    }


    public function store( Request $request, $id ){
        if($id>0){
            $this->checkPermission('course_update');
            $row = $this->courseClass::find($id);
            if (empty($row)) {
                return redirect(route('course.vendor.index'));
            }

            if($row->create_user != Auth::id() and !$this->hasPermission('course_manage_others'))
            {
                return redirect(route('course.vendor.index'));
            }
        }else{
            $this->checkPermission('course_create');
            $row = new $this->courseClass();
            $row->status = "publish";
            if(setting_item("course_vendor_create_service_must_approved_by_admin", 0)){
                $row->status = "pending";
            }
        }
        $dataKeys = [
            'title',
            'content',
            'price',
            'is_instant',
            'video',
            'faqs',
            'image_id',
            'banner_image_id',
            'gallery',
            'location_id',
            'address',
            'map_lat',
            'map_lng',
            'map_zoom',
            'number',
            'price',
            'original_price',
            'passenger',
            'gear',
            'baggage',
            'door',
            'enable_extra_price',
            'extra_price',
            'is_featured',
            'default_state',
        ];
        if($this->hasPermission('course_manage_others')){
            $dataKeys[] = 'author_id';
        }

        $row->fillByAttr($dataKeys,$request->input());

        $res = $row->saveOriginOrTranslation($request->input('lang'),true);

        if ($res) {
            if(!$request->input('lang') or is_default_lang($request->input('lang'))) {
                $this->saveTerms($row, $request);
            }

            if($id > 0 ){
                return back()->with('success',  __('Course updated') );
            }else{
                return redirect(route('course.vendor.edit',['id'=>$row->id]))->with('success', __('Course created') );
            }
        }
    }

    public function saveTerms($row, $request)
    {
        if (empty($request->input('terms'))) {
            $this->jobTermClass::where('target_id', $row->id)->delete();
        } else {
            $term_ids = $request->input('terms');
            foreach ($term_ids as $term_id) {
                $this->jobTermClass::firstOrCreate([
                    'term_id' => $term_id,
                    'target_id' => $row->id
                ]);
            }
            $this->jobTermClass::where('target_id', $row->id)->whereNotIn('term_id', $term_ids)->delete();
        }
    }

    public function editCourse(Request $request, $id)
    {
        $this->checkPermission('course_update');
        $user_id = Auth::id();
        $row = $this->courseClass::where("create_user", $user_id);
        $row = $row->find($id);
        if (empty($row)) {
            return redirect(route('course.vendor.index'))->with('warning', __('Course not found!'));
        }
        $translation = $row->translate($request->query('lang'));
        $data = [
            'translation'    => $translation,
            'row'           => $row,
            'course_location' => $this->locationClass::get()->toTree(),
            'attributes'    => $this->attributesClass::where('service', 'course')->get(),
            "selected_terms" => $row->terms->pluck('term_id'),
            'breadcrumbs'        => [
                [
                    'name' => __('Manage Courses'),
                    'url'  => route('course.vendor.index')
                ],
                [
                    'name'  => __('Edit'),
                    'class' => 'active'
                ],
            ],
            'page_title'         => __("Edit Courses"),
        ];
        return view('Course::frontend.manageCourse.detail', $data);
    }

    public function deleteCourse($id)
    {
        $this->checkPermission('course_delete');
        $user_id = Auth::id();
        $query = $this->courseClass::where("create_user", $user_id)->where("id", $id)->first();
        if(!empty($query)){
            $query->delete();
        }
        return redirect(route('course.vendor.index'))->with('success', __('Delete car success!'));
    }

    public function bulkEditCourse($id , Request $request){
        $this->checkPermission('course_update');
        $action = $request->input('action');
        $user_id = Auth::id();
        $query = $this->courseClass::where("create_user", $user_id)->where("id", $id)->first();
        if (empty($id)) {
            return redirect()->back()->with('error', __('No item!'));
        }
        if (empty($action)) {
            return redirect()->back()->with('error', __('Please select an action!'));
        }
        if(empty($query)){
            return redirect()->back()->with('error', __('Not Found'));
        }
        switch ($action){
            case "make-hide":
                $query->status = "draft";
                break;
            case "make-publish":
                $query->status = "publish";
                break;
        }
        $query->save();
        return redirect()->back()->with('success', __('Update success!'));
    }

    public function bookingReport(Request $request)
    {
        $data = [
            'bookings' => $this->bookingClass::getBookingHistory($request->input('status'), false , Auth::id() , 'course'),
            'statues'  => config('booking.statuses'),
            'breadcrumbs'        => [
                [
                    'name' => __('Manage Course'),
                    'url'  => route('course.vendor.index')
                ],
                [
                    'name' => __('Booking Report'),
                    'class'  => 'active'
                ]
            ],
            'page_title'         => __("Booking Report"),
        ];
        return view('Course::frontend.manageCourse.bookingReport', $data);
    }

    public function bookingReportBulkEdit($booking_id , Request $request){
        $status = $request->input('status');
        if (!empty(setting_item("course_allow_vendor_can_change_their_booking_status")) and !empty($status) and !empty($booking_id)) {
            $query = $this->bookingClass::where("id", $booking_id);
            $query->where("vendor_id", Auth::id());
            $item = $query->first();
            if(!empty($item)){
                $item->status = $status;
                $item->save();
                $item->sendStatusUpdatedEmails();
                return redirect()->back()->with('success', __('Update success'));
            }
            return redirect()->back()->with('error', __('Booking not found!'));
        }
        return redirect()->back()->with('error', __('Update fail!'));
    }
}
